<?php

namespace App\Notifications;

use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use NotificationChannels\Fcm\FcmChannel;
use NotificationChannels\Fcm\FcmMessage;
use NotificationChannels\Fcm\Resources\Notification as FcmNotification;

class NewOrderProviderNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $item;
    protected $user;

    public function __construct(BookingItem $item, User $user)
    {
        $this->item = $item;
        $this->user = $user;
    }

    public function via($notifiable)
    {
        $channels = ['database'];

        if ($notifiable->firebase_token) {
            $channels[] = FcmChannel::class;
        }

        return $channels;
    }

    public function toFcm($notifiable): FcmMessage
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();
        $title = __('messages.new_order_title', ['user' => $this->user->name], $locale);
        $body = __('messages.new_order_body', ['item_id' => $this->item->id, 'price' => $this->item->price, 'date' => $this->item->service_date], $locale);

        return FcmMessage::create()
            ->data([
                'item_id' => (string) $this->item->id,
                'url' => route('provider.booking-items.show', $this->item->id),
                'type' => 'new_order_item',
            ])
            ->notification(
                notification: FcmNotification::create()
                    ->title($title)
                    ->body($body)
            );
    }

    public function toArray($notifiable)
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();

        return [
            'title_en' => __('messages.new_order_title', ['user' => $this->user->name], 'en'),
            'body_en' => __('messages.new_order_body', ['item_id' => $this->item->id, 'price' => $this->item->price, 'date' => $this->item->service_date], 'en'),
            'title_ar' => __('messages.new_order_title', ['user' => $this->user->name], 'ar'),
            'body_ar' => __('messages.new_order_body', ['item_id' => $this->item->id, 'price' => $this->item->price, 'date' => $this->item->service_date], 'ar'),
            'url' => route('provider.booking-items.show', $this->item->id),
        ];
    }
}
