<?php

namespace App\Notifications;

use App\Models\BookingItem;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use NotificationChannels\Fcm\FcmChannel;
use NotificationChannels\Fcm\FcmMessage;
use NotificationChannels\Fcm\Resources\Notification as FcmNotification;

class ItemCancelledNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $item;
    protected $user;

    public function __construct(BookingItem $item, User $user)
    {
        $this->item = $item;
        $this->user = $user;
    }

    public function via($notifiable)
    {
        $channels = ['database'];

        if ($notifiable->firebase_token) {
            $channels[] = FcmChannel::class;
        }

        return $channels;
    }

    public function toFcm($notifiable): FcmMessage
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();
        $title = __('messages.item_cancelled_title', ['product' => $this->item->product->name ?? 'Product'], $locale);
        $body = __('messages.item_cancelled_body', ['item_id' => $this->item->id, 'date' => $this->item->service_date], $locale);

        return FcmMessage::create()
            ->data([
                'item_id' => (string) $this->item->id,
                'url' => route('provider.booking-items.show', $this->item->id),
                'booking_id' => $this->item->booking_id,
                'type' => 'item_cancelled',

            ])
            ->notification(
                notification: FcmNotification::create()
                    ->title($title)
                    ->body($body)
            );
    }

    public function toArray($notifiable)
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();

        return [
            'title_en' => __('messages.item_cancelled_title', ['product' => $this->item->product->name ?? 'Product'], 'en'),
            'body_en' => __('messages.item_cancelled_body', ['item_id' => $this->item->id, 'date' => $this->item->service_date], 'en'),
            'title_ar' => __('messages.item_cancelled_title', ['product' => $this->item->product->name ?? 'Product'], 'ar'),
            'body_ar' => __('messages.item_cancelled_body', ['item_id' => $this->item->id, 'date' => $this->item->service_date], 'ar'),
            'url' => route('provider.booking-items.show', $this->item->id),
            'booking_id' => $this->item->booking_id,
        ];
    }
}