<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use App\Models\Booking;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use NotificationChannels\Fcm\FcmChannel;
use NotificationChannels\Fcm\FcmMessage;
use NotificationChannels\Fcm\Resources\Notification as FcmNotification;

class BookingStatusUpdatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(public Booking $booking)
    {
        //
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];

        if ($notifiable->firebase_token) {
            $channels[] = FcmChannel::class;
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->line('The introduction to the notification.')
            ->action('Notification Action', url('/'))
            ->line('Thank you for using our application!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'booking_id' => $this->booking->id,
            'title_en' => __('messages.booking_status_updated_title', ['status' => $this->booking->status->label()], 'en'),
            'body_en' => __('messages.booking_status_updated_body', ['id' => $this->booking->id, 'status' => $this->booking->status->label()], 'en'),
            'title_ar' => __('messages.booking_status_updated_title', ['status' => $this->booking->status->label()], 'ar'),
            'body_ar' => __('messages.booking_status_updated_body', ['id' => $this->booking->id, 'status' => $this->booking->status->label()], 'ar'),
            'url' => route('user.bookings.show', $this->booking->id),
        ];
    }

    public function toFcm($notifiable): FcmMessage
    {
        $locale = $notifiable->pref_lang ?? app()->getLocale();
        $title = __('messages.booking_status_updated_title', ['status' => $this->booking->status->label()], $locale);
        $body = __('messages.booking_status_updated_body', ['id' => $this->booking->id, 'status' => $this->booking->status->label()], $locale);

        return FcmMessage::create()
            ->data([
                'booking_id' => (string) $this->booking->id,
                'url' => route('user.bookings.show', $this->booking->id),
            ])
            ->notification(
                notification: FcmNotification::create()
                    ->title($title)
                    ->body($body)
            );
    }
}
