<?php

namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\Traits\ApiResponse;
use App\Http\Resources\ReviewResource;
use App\Models\Review;
use App\Http\Requests\Api\User\StoreReviewRequest;
use Illuminate\Support\Facades\Auth;

class ReviewController extends Controller
{
    use ApiResponse;

    /**
     * @OA\Post(
     *     path="/api/user/reviews",
     *     summary="Submit a review for a product",
     *     tags={"Reviews"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"product_id", "rating", "comment"},
     *             @OA\Property(property="product_id", type="integer", example=1),
     *             @OA\Property(property="rating", type="integer", example=5),
     *             @OA\Property(property="comment", type="string", example="Great product!")
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Review submitted successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/ReviewResource"),
     *             @OA\Property(property="message", type="string", example="Review submitted successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error"
     *     )
     * )
     */
    public function store(StoreReviewRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $review_data = $request->validated();
        $product_id = $review_data['product_id'];
        $review = $user->reviews()->updateOrCreate(['product_id' => $product_id], $review_data);

        return $this->successResponse(new ReviewResource($review), __('messages.review_submitted_successfully'), 201);
    }
}
