<?php

namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\Traits\ApiResponse;
use App\Http\Requests\Api\User\StoreFavoriteRequest;
use App\Http\Resources\FavoriteResource;
use Illuminate\Support\Facades\Auth;
use OpenApi\Annotations as OA;

class FavoriteController extends Controller
{
    use ApiResponse;

    /**
     * @OA\Get(
     *     path="/api/user/favorites",
     *     summary="Get the user's favorites",
     *     tags={"Favorites"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Parameter(
     *         name="type",
     *         in="query",
     *         required=false,
     *         description="type of product (service,package)",
     *         @OA\Schema(type="string", enum={"service", "package"})
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="The user's favorite items",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="array", @OA\Items(ref="#/components/schemas/FavoriteResource")),
     *             @OA\Property(property="message", type="string", example="Favorites retrieved successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     )
     * )
     */
    public function index()
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $favorites = $user->favorites()
            ->when(request()->input('type'), function ($query) {
                $query->where("type", request()->input('type'));
            })
            ->with('product.media')->get();
        return $this->successResponse(FavoriteResource::collection($favorites), __('messages.favorites_retrieved_successfully'));
    }

    /**
     * @OA\Post(
     *     path="/api/user/favorites",
     *     summary="Add an item to favorites",
     *     tags={"Favorites"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"product_id"},
     *             @OA\Property(property="product_id", type="integer", example=1)
     *         )
     *     ),
     *     @OA\Response(
     *         response=201,
     *         description="Item added to favorites successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/FavoriteResource"),
     *             @OA\Property(property="message", type="string", example="Item added to favorites successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error"
     *     )
     * )
     */
    public function store(StoreFavoriteRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $favorite = $user->favorites()->updateOrCreate($request->validated());

        return $this->successResponse(new FavoriteResource($favorite), __('messages.item_added_to_favorites'), 201);
    }

    /**
     * @OA\Delete(
     *     path="/api/user/favorites/{id}",
     *     summary="Remove an item from favorites",
     *     tags={"Favorites"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Parameter(
     *         name="id",
     *         in="path",
     *         required=true,
     *         description="Favorite item ID",
     *         @OA\Schema(type="integer")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Item removed from favorites successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="object", example=null),
     *             @OA\Property(property="message", type="string", example="Item removed from favorites successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=404,
     *         description="Favorite item not found"
     *     )
     * )
     */
    public function destroy($id)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
        $favorite = $user->favorites()->findOrFail($id);
        $favorite->delete();

        return $this->successResponse(null, __('messages.item_removed_from_favorites'), 200);
    }
}
