<?php

namespace App\Http\Controllers\Api\Provider;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Provider\ChangePasswordRequest;
use App\Http\Requests\Api\Provider\UpdateBankDetailsRequest;
use App\Http\Controllers\Api\Traits\ApiResponse;
use App\Http\Requests\Api\Provider\UpdateProfileRequest;
use App\Http\Resources\ServiceProviderResource;
use App\Services\ProviderProfileService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use OpenApi\Annotations as OA;

class ProfileController extends Controller
{
    use ApiResponse;
    protected $providerProfileService;

    public function __construct(ProviderProfileService $providerProfileService)
    {
        $this->providerProfileService = $providerProfileService;
    }
    /**
     * @OA\Get(
     *     path="/api/provider/profile",
     *     summary="Get the authenticated provider's profile",
     *     tags={"Provider Profile"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Provider profile data",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/ServiceProviderResource"),
     *             @OA\Property(property="message", type="string", example="Profile retrieved successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     )
     * )
     */
    public function show()
    {
        return $this->successResponse(new ServiceProviderResource($this->providerProfileService->getProfile()), __('messages.profile_retrieved_successfully'));
    }

    /**
     * @OA\Post(
     *     path="/api/provider/profile",
     *     summary="Update the authenticated provider's profile",
     *     tags={"Provider Profile"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"category_id", "name", "email", "phone", "nationality_id", "cities"},
     *                 @OA\Property(property="category_id", type="integer", example=1, description="ID of the service category"),
     *                 @OA\Property(property="name", type="string", example="New Provider Name", minLength=4, maxLength=100),
     *                 @OA\Property(property="nationality_id", type="integer", example=1, description="ID of the nationality"),
     *                 @OA\Property(property="cities[]", type="array", @OA\Items(type="integer", example=1), minItems=1, description="Array of city IDs the provider operates in"),
     *                 @OA\Property(property="logo", type="string", format="binary", description="Profile picture file (e.g., image.jpg)")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Provider profile updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/ServiceProviderResource"),
     *             @OA\Property(property="message", type="string", example="Profile updated successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error"
     *     )
     * )
     */
    public function update(UpdateProfileRequest $request)
    {
        $provider = $this->providerProfileService->updateProfile($request->validated());

        return $this->successResponse(new ServiceProviderResource($provider), __('messages.profile_updated'));
    }

    /**
     * @OA\Put(
     *     path="/api/provider/bank-details",
     *     summary="Update the authenticated provider's bank details",
     *     tags={"Provider Profile"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="iban", type="string", example="SA0380000000608010167519"),
     *             @OA\Property(property="bank_name", type="string", example="Al-Rajhi Bank")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Bank details updated successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", ref="#/components/schemas/ServiceProviderResource"),
     *             @OA\Property(property="message", type="string", example="Bank details updated successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error"
     *     )
     * )
     */
    public function updateBankDetails(UpdateBankDetailsRequest $request)
    {
        $provider = $this->providerProfileService->updateBankDetails($request->validated());

        return $this->successResponse(new ServiceProviderResource($provider), __('messages.bank_details_updated'));
    }

    /**
     * @OA\Put(
     *     path="/api/provider/password/change",
     *     summary="Change the authenticated provider's password",
     *     tags={"Provider Profile"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="password", type="string", format="password"),
     *             @OA\Property(property="new_password", type="string", format="password"),
     *             @OA\Property(property="new_password_confirmation", type="string", format="password")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Password changed successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="object", example=null),
     *             @OA\Property(property="message", type="string", example="Password changed successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     ),
     *     @OA\Response(
     *         response=422,
     *         description="Validation error"
     *     )
     * )
     */
    public function changePassword(ChangePasswordRequest $request)
    {
        $success = $this->providerProfileService->changePassword($request->validated());

        if (!$success) {
            return $this->errorResponse(__('messages.current_password_does_not_match'), 422);
        }

        return $this->successResponse(null, __('messages.password_changed'));
    }

    /**
     * @OA\Delete(
     *     path="/api/provider/profile",
     *     summary="Delete the authenticated provider's account",
     *     tags={"Provider Profile"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *         name="Accept-Language",
     *         in="header",
     *         required=false,
     *         description="Language preference (e.g., 'en', 'ar')",
     *         @OA\Schema(type="string", enum={"en", "ar"}, default="en")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Account deleted successfully",
     *         @OA\JsonContent(
     *             @OA\Property(property="success", type="boolean", example=true),
     *             @OA\Property(property="data", type="object", example=null),
     *             @OA\Property(property="message", type="string", example="Account deleted successfully.")
     *         )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated"
     *     )
     * )
     */
    public function destroy()
    {
        if (auth()->user()->upcomingBookingItems()->exists()) {
            return $this->errorResponse(__('messages.cannot_delete_account_with_upcoming_bookings'), 422);
        }

        $this->providerProfileService->deleteProfile();

        return $this->successResponse(null, __('messages.account_deleted_successfully'));
    }
}
